<?php

namespace App\Http\Controllers\Admin;

use App\Models\Role;
use App\Models\User;
use Illuminate\Http\Request;
use App\Mail\UserStatusChanged;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Mail;
use App\Http\Requests\UserStoreRequest;
use App\Http\Requests\UserUpdateRequest;

class UserController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        //authorize this user to access/give access to admin dashboard
        Gate::authorize('index-user');

        $users = User::with(['role:id,role_name,role_slug', 'profile', 'adminProfileImage'])
        ->select(['id', 'role_id', 'department_id', 'name', 'email', 'is_active', 'updated_at'])
        ->paginate(1000);

        $roles = Role::where('is_deletable', 1)->select(['id', 'role_name'])->get();

        return view('admin.user.index', compact('users', 'roles'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(UserStoreRequest $request)
    {
        //authorize this user to access/give access to admin dashboard
        Gate::authorize('create-user');

        // dd($request->all());
        User::create([
            'role_id' => $request->role_id,
            'name' => $request->name,
            'email' => $request->email,
            'password' => Hash::make($request->password)
        ]);

        return redirect()->back()->with('message', 'User Created Successfully 🙂');

    }

    /**
     * Display the specified resource.
     */
    public function show($id)
    {
        //authorize this user to access/give access to admin dashboard
        Gate::authorize('edit-user');

        $user = User::with(['role:id,role_name'])->where('id', $id)->first();
        // $roles = Role::select(['id', 'role_name'])->get();

        return view('admin.user.view', compact('user'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit($id)
    {
        //authorize this user to access/give access to admin dashboard
        Gate::authorize('edit-user');

        $user = User::where('id', $id)->first();
        $roles = Role::select(['id', 'role_name'])->get();

        return view('admin.user.edit', compact('user', 'roles'));

    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, $id)
    {
        //authorize this user to access/give access to admin dashboard
        Gate::authorize('edit-user');

        // dd($request->all(), $id);
        $user = User::where('id', $id)->first();

        $user->update([
            'is_active' => $request->filled('is_active'),
            'is_merchandiser' => $request->filled('is_merchandiser'),
        ]);

        // Send confirmation mail if the user is a merchandiser
        if ($user->is_merchandiser) {
            Mail::to($user->email)->send(new UserStatusChanged($user, true));
        } else {
            Mail::to($user->email)->send(new UserStatusChanged($user, false));
        }

        return redirect()->route('users.index')->with('message', 'User Updated Successfully 🙂');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        //authorize this user to access/give access to admin dashboard
        Gate::authorize('delete-user');

        // dd($id);
        $user = User::where('id', $id)->first();
        if ($user->email != 'admin@admin.com') {
            $user->delete();

            return redirect()->back()->with('warning', 'User Deleted Successfully');
        } else {
            return redirect()->back()->with('error', 'Admin Cannot be Deleted 😡!!');
        }



    }
}
